function [lambda p gof] = expFitDeg(obj, varargin)
%EXPFITDEG Fit the degree of this matrix to an exponential distribution
%   [h p gof] = PLFITOUTDEG(obj, ... param, paramValue, ... ) fits the
%   degree of AdjacencyMatrix obj to an exponential distribution.  This
%   calls the function expFit. The return values are:
%       h - a pair of handles to the plotted data sets (data and fitted
%           data)
%       p - p-value for this fit (according to the heurisitic in Clauset
%           et. al. - a p-value less than .1 indicates that the exponential
%           distribution has low probability of generating a data set with
%           as poor a K-S statistic as this data set.
%       gof - Kolmogorov goodness of fit statistic for this fit
%
%   Optional parameters:
%       'degreeType' - should be 'in', 'out', or 'total'.  Default is
%           'total'.  Determines the type of degree to be fit (in-degree,
%           out-degree, or total degree)
%       'weighted' - should be true or false.  Determines whether edge
%           weights are considered.  Default is true.
%       'tolerance' - if 'weighted' is set to false, then the
%           tolerance will determine the lower limit for the weight of an
%           edge. Default is 1.  Set to a higher number to eliminate noise;
%           spurious edges.
%       'pTest' - if set to 'false', this will skip the p-value
%           calculation.  This calculation is time-intensive, so set this
%           parameter to speed up the function.  Default is 'true'.

weighted = true;
degreeType = 'total';
tolerance = 1;
calcPVal = true;

hld = false;
dataLabels = 'bo';

for k = 1:2:length(varargin)
    switch varargin{k}
        case 'degreeType'
            degreeType = varargin{k+1};
        case 'tolerance'
            tolerance = varargin{k+1};
        case 'weighted'
            weighted = varargin{k+1};
        case 'pTest'
            calcPVal = varargin{k+1};
        case 'hold'
            hld = varargin{k+1};
        case 'dataLabels'
            dataLabels = varargin{k+1};
    end
end

switch weighted
    case true
        A = obj.A;
    case false
        A = obj.A;
        A = double(A>= tolerance);
end


switch degreeType
    case 'in'
        x = sum(A,1);
    case 'out'
        x = sum(A,2);
    case 'total'
        switch weighted
            case true
                A = getSquareMatrix(obj);
                colSums = sum(A,1);
                rowSums = sum(A,2);
                x = rowSums + colSums';
            case false
                A = getSquareMatrix(obj);
                A = double(A >= tolerance);
                colSums = sum(A,1);
                rowSums = sum(A,2);
                x = rowSums + colSums';
        end
end

[lambda,xmin,L] = expFit(x);

% plot the results
h = expPlot(x,xmin,lambda,'hold',hld,'dataLabels',dataLabels);
if weighted, w = 'weighted'; else w = 'unweighted'; end
title(gca,sprintf('Exponential fit of %s %s degree\nlambda = %.5f',w,degreeType,lambda));
set(gcf,'Color',[1 1 1]);
xlabel('x');
ylabel('P(degree > x)');
fprintf('Xmin: %.0f\n',xmin);
fprintf('lambda: %.5f\n',lambda);

p = 0;
gof = 0;
if calcPVal
    [p gof] = expFitPVa(x);
    fprintf('p-value: %.5f\n',p);
    fprintf('Kolmogorov-Smirnov goodness of fit statistic: %.5f\n',gof);
end

end

